/************************************************************************
 * @file: AudioFactory.cpp
 *
 * @version: 0.1
 *
 * @description: This source file contains implementation for class Factory.
 * Factory class will be used by application to select the Backend Libraries,
 * based on the Backend Name [eg: Alsa] provided by application for
 * Factory class. This class provides handle for the selected Backend to
 * Application.
 *
 * @authors: Jens Lorenz, jlorenz@de.adit-jv.com 2015
 *           Thouseef Ahamed, tahamed@de.adit-jv.com 2015
 *           Vijay Palaniswamy, vijay.palaniswamy@in.bosch.com 2015
 *
 * @copyright (c) 2015 Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 ***********************************************************************/

#include <dirent.h>
#include <dlfcn.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <cstring>
#include <algorithm>

#include "AudioFactory.h"
#include "AudioBackendDefault.h"
#include "Logging.h"


#define ARCH_TO_STRING(macro) TOKENIZE(macro)
#define TOKENIZE(LIB_SUFFIX) #LIB_SUFFIX

#ifndef BE_LIBRARY_PATH
#define BE_LIBRARY_PATH   "/usr/lib" ARCH_TO_STRING(LIB_SUFFIX) "/audiobe"
#endif

using namespace std;
using namespace adit::utility::audio;
using namespace adit::utility;

Factory* Factory::singleton = nullptr;

Factory::Factory(LoggingInterface& loggingHandle) : mLoggingHandle(loggingHandle)
{
}

Factory* Factory::Instance(LoggingInterface& loggingHandle)
{
    if (singleton == nullptr)
    {
        singleton = new Factory(loggingHandle);
    }
    return singleton;
}

Factory* Factory::Instance()
{
    /* Default logging class */
    class FactoryLogging : public LoggingInterface
    {
        #define TEXT_COLOR_RED    "\33[0;31m"
        #define TEXT_COLOR_YELLOW "\033[0;33m"
        #define TEXT_COLOR_RESET  "\033[0m"

        void error(const string& data) const final
        {
            cerr << TEXT_COLOR_RED << data << TEXT_COLOR_RESET << endl;
        }

        void warning(const string& data) const final
        {
            cerr << TEXT_COLOR_YELLOW << data << TEXT_COLOR_RESET << endl;
        }

        void info(const string& data) const final
        {
            cout << data << endl;
        }

        void debug(const string& data) const final
        {
            cout << data << endl;
        }

        eLogLevel checkLogLevel() const final
        {
            return LL_ERROR;
        }
    } static FactoryLog;

    return Instance(FactoryLog);
}

Factory::~Factory(void)
{
}

AudioError Factory::getSupportedBe(vector<string>& backends)
{
    AudioError ret = AudioError::OK;
    DIR *directory = opendir(BE_LIBRARY_PATH);
    if (!directory)
    {
        Logging(mLoggingHandle, LL_ERROR) << "Factory::getSupportedBe Error opening directory: " << BE_LIBRARY_PATH << Logging::endl;
        return AudioError::FAILURE;
    }

    struct dirent *entry = NULL;
    while ((entry = readdir(directory)))
    {
        string name = entry->d_name;
        string fullName = BE_LIBRARY_PATH;
        fullName += "/" + name;

        bool regularFile = (entry->d_type == DT_REG || entry->d_type == DT_LNK);
        if (entry->d_type == DT_UNKNOWN)
        {
            struct stat buf;

            if (stat(fullName.c_str(), &buf))
            {
                Logging(mLoggingHandle, LL_ERROR) << "Factory::getSupportedBe Failed to stat file: " << name << " " << strerror(errno) << Logging::endl;
                continue;
            }

            regularFile = S_ISREG(buf.st_mode);
        }

        if (regularFile && ("so" == name.substr(name.find_last_of(".") + 1)))
        {
            /* discard "lib" in front and ".so" at the end */
            backends.push_back(name.substr(3, name.length() - 6));
            {
                string entry = name.substr(3, name.length() - 6);
                Logging(mLoggingHandle, LL_INFO) << "Factory::getSupportedBe Adding lib " << entry << Logging::endl;
            }
        }
        else
        {
            Logging(mLoggingHandle, LL_INFO) << "Factory::getSupportedBe Ignoring backend library: " << name << Logging::endl;
        }
    }

    closedir(directory);

    return ret;
}

shared_ptr<Backend> Factory::createBackend(string& backend, Streaming& streamingHandle)
{
    /* If Backend Name is empty, In this case we try to open the first Backend found in respective module folder.
     * If nothing is found the default Backend will be selected
     */
    if (backend.empty())
    {
        vector<string> backends;
        getSupportedBe(backends);
        if (backends.size())
        {
            backend = backends[0];
        }
        else
        {
            Logging(mLoggingHandle, LL_WARNING) << "Load default plug-in" << Logging::endl;
            return shared_ptr<Backend>(new BackendDefault());
        }
    }

    /* Create library name */
    string libname = BE_LIBRARY_PATH"/lib" + backend + ".so";

    Logging(mLoggingHandle, LL_DEBUG) << "Factory::createBackend Trying to load library with name: " << libname << Logging::endl;

    dlerror(); /* To flush Error string */

    /* dlopen library */
    void* soLib = dlopen(libname.c_str(), RTLD_LAZY);
    const char* dlopen_error = dlerror();
    if (!soLib || dlopen_error)
    {
        Logging(mLoggingHandle, LL_ERROR) << "Factory::createBackend " << dlopen_error << Logging::endl;
        return nullptr;
    }

    Logging(mLoggingHandle, LL_DEBUG) << "Factory::createBackend dlopen is success" << Logging::endl;

    /* Create entry function name */
    string beEntryFunction = backend + "BeEntry";

    /* get entry point from shared lib */
    typedef Backend* (*CFunc)(Streaming&);
    union {
        void* typeless;
        CFunc typespec;
    } ptr_cast;

    ptr_cast.typeless = dlsym(soLib, beEntryFunction.c_str());
    const char* dlsym_error = dlerror();
    if (!ptr_cast.typeless || dlsym_error)
    {
        Logging(mLoggingHandle, LL_ERROR) << "Factory::createBackend " << dlsym_error << Logging::endl;
        dlclose(soLib);
        return nullptr;
    }

    Logging(mLoggingHandle, LL_DEBUG) << "Factory::createBackend dlsym is success" << Logging::endl;

    Backend* audioBE = ptr_cast.typespec(streamingHandle);
    if (!audioBE)
    {
        Logging(mLoggingHandle, LL_ERROR) << "Factory::createBackend Backend Initialization failed" << Logging::endl;
        dlclose(soLib);
        return nullptr;
    }

    Logging(mLoggingHandle, LL_INFO) << "Factory::createBackend Loaded Backend \"" << backend << "\" successfully" << Logging::endl;
    return shared_ptr<Backend>(audioBE, [=](Backend *p) { delete p; dlclose (soLib); });
}
